function get_dependencies_dir()
{
    $DependencyPackagesDir = $null
    try
    {
        $ScriptPath = $PSScriptRoot
        $DependencyPackagesDir = Join-Path -Path $ScriptPath -ChildPath "Dependencies"
    }
    catch {
        Write-Host "Error occured: unable to get dependency directory"
    }

    return $DependencyPackagesDir
}

function is_admin()
{
    # Check if the current user has administrative rights
    $currentUser = [Security.Principal.WindowsIdentity]::GetCurrent()
    $principal = New-Object Security.Principal.WindowsPrincipal($currentUser)
    return $principal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
}

function get_current_username()
{
    return $(Get-WMIObject -class Win32_ComputerSystem | select username).username
}

function get_dependencies()
{
    $ProcessorArchitecture = $Env:Processor_Architecture
    if ($ProcessorArchitecture -eq "x86")
    {
        if ($null -ne ${Env:ProgramFiles(Arm)})
        {
            $ProcessorArchitecture = "arm64"
        }
        elseif ($null -ne ${Env:ProgramFiles(x86)})
        {
            $ProcessorArchitecture = "amd64"
        }
    }
    Write-Host "> Architecture: $ProcessorArchitecture"

    $DependencyPackagesDir = get_dependencies_dir
    Write-Host "> Dependencies dir: $DependencyPackagesDir"

    $DependencyPackages = @()
    if (Test-Path $DependencyPackagesDir)
    {
        # Get architecture-neutral dependencies
        $DependencyPackages += Get-ChildItem (Join-Path $DependencyPackagesDir "*.appx") | Where-Object { $_.Mode -NotMatch "d" }
        $DependencyPackages += Get-ChildItem (Join-Path $DependencyPackagesDir "*.msix") | Where-Object { $_.Mode -NotMatch "d" }

        # Get architecture-specific dependencies
        if (($ProcessorArchitecture -eq "x86" -or $ProcessorArchitecture -eq "amd64" -or $ProcessorArchitecture -eq "arm64") -and (Test-Path (Join-Path $DependencyPackagesDir "x86")))
        {
            $DependencyPackages += Get-ChildItem (Join-Path $DependencyPackagesDir "x86\*.appx") | Where-Object { $_.Mode -NotMatch "d" }
            $DependencyPackages += Get-ChildItem (Join-Path $DependencyPackagesDir "x86\*.msix") | Where-Object { $_.Mode -NotMatch "d" }
        }
        if (($ProcessorArchitecture -eq "amd64") -and (Test-Path (Join-Path $DependencyPackagesDir "x64")))
        {
            $DependencyPackages += Get-ChildItem (Join-Path $DependencyPackagesDir "x64\*.appx") | Where-Object { $_.Mode -NotMatch "d" }
            $DependencyPackages += Get-ChildItem (Join-Path $DependencyPackagesDir "x64\*.msix") | Where-Object { $_.Mode -NotMatch "d" }
        }
        if (($ProcessorArchitecture -eq "arm" -or $ProcessorArchitecture -eq "arm64") -and (Test-Path (Join-Path $DependencyPackagesDir "arm")))
        {
            $DependencyPackages += Get-ChildItem (Join-Path $DependencyPackagesDir "arm\*.appx") | Where-Object { $_.Mode -NotMatch "d" }
            $DependencyPackages += Get-ChildItem (Join-Path $DependencyPackagesDir "arm\*.msix") | Where-Object { $_.Mode -NotMatch "d" }
        }
        if (($ProcessorArchitecture -eq "arm64") -and (Test-Path (Join-Path $DependencyPackagesDir "arm64")))
        {
            $DependencyPackages += Get-ChildItem (Join-Path $DependencyPackagesDir "arm64\*.appx") | Where-Object { $_.Mode -NotMatch "d" }
            $DependencyPackages += Get-ChildItem (Join-Path $DependencyPackagesDir "arm64\*.msix") | Where-Object { $_.Mode -NotMatch "d" }
        }
    }
    if ($DependencyPackages.Count -eq 0)
    {
        Write-Host "> Error: Unable to find dependency packages"
        exit 1
    }
    Write-Host "> Dependencies are: $DependencyPackages"
    return $DependencyPackages
}

Function install($package) {
    $packages = Get-AppxPackage -Name "3D4Medical.comLLC.CompleteAnatomy"
    if ($null -ne $packages) {
        Write-Host "> Error: Appx package already installed, uninstall it first"
        exit 1
    }
    $dependencies = get_dependencies
    $logName = "install_$(Get-Date -f 'yyyyMMddhhmmss').log"
    $logsPath = "$PSScriptRoot\$logName"
    Add-AppxPackage -Path $package -DependencyPath $dependencies.FullName -ForceApplicationShutdown *>> $logsPath
    $packages = Get-AppxPackage -Name "3D4Medical.comLLC.CompleteAnatomy"
    if ($null -eq $packages) {
        Write-Host "> Error: Unable to find Appx package, retry installation"
        Add-AppxPackage -Path $package -DependencyPath $dependencies.FullName -ForceApplicationShutdown *>> $logsPath
    }
}

Function uninstall() {
    $logName = "uninstall_$(Get-Date -f 'yyyyMMddhhmmss').log"
    $logsPath = "$PSScriptRoot\$logName"
    $packages = Get-AppxPackage -Name "3D4Medical.comLLC.CompleteAnatomy"
    if ($null -eq $packages) {
        Write-Host "> Error: Unable to find Appx package, probablhy that is not installed"
        exit 1
    }
    foreach ($package in $packages)
    {
        Write-Host "> Uninstalling package $package"
        Remove-AppxPackage -Package $package *>> $logsPath
    }
}

Function uninstall_dism() {
    if (-not (is_admin)) {
        Write-Host "> Error: This script must be run as an administrator"
        return
    }

    Get-AppxProvisionedPackage -Online | Where-Object {$_.packagename -like "*CompleteAnatomy*"} | Remove-AppxProvisionedPackage -online
    $logName = "uninstall_$(Get-Date -f 'yyyyMMddhhmmss').log"
    $packages = Get-AppxPackage -AllUsers -Name "3D4Medical.comLLC.CompleteAnatomy"
    if ($null -eq $packages) {
        Write-Host "> Error: Unable to find Appx package, probablhy that is not installed"
        return
    }
    foreach ($package in $packages)
    {
        Write-Host "> Uninstalling package $package"
        Remove-AppxPackage -AllUsers -Package $package
    }
}

Function install_dism($package, $license) {
    if (-not (is_admin)) {
        Write-Host "> Error: This script must be run as an administrator"
        return
    }

    $dependencies = get_dependencies
    $dependenciesParam = ($dependencies | ForEach-Object { "/DependencyPackagePath:`"$($_.FullName)`"" }) -join " "
    $cmd = "Dism /Online /Add-ProvisionedAppxPackage /PackagePath:`"$package`" $dependenciesParam /LicensePath:`"$license`""
    Invoke-Expression $cmd
}